/*
 * Copyright (C) 2009-2012 University of Freiburg
 *
 * This file is part of SMTInterpol.
 *
 * SMTInterpol is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * SMTInterpol is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SMTInterpol.  If not, see <http://www.gnu.org/licenses/>.
 */
/* SMT-Lib lexer */
package de.uni_freiburg.informatik.ultimate.smtinterpol.smtlib;
import java_cup.runtime.Symbol;
import de.uni_freiburg.informatik.ultimate.smtinterpol.util.MySymbolFactory;

/**
 * This is a autogenerated lexer for smtlib.
 * It is generated from smtlib.flex by JFlex.
 */
%%

%class Lexer
%public
%unicode
%cupsym LexerSymbols
%cup
%line
%column

%{
  private StringBuffer string = new StringBuffer();
  private MySymbolFactory symFactory;
  
  public void setSymbolFactory(MySymbolFactory factory) {
    symFactory = factory;
  }

  private Symbol symbol(int type) {
    return symFactory.newSymbol(yytext(), type, yyline+1, yycolumn, yyline+1, yycolumn+yylength());
  }
  private Symbol symbol(int type, String value) {
    return symFactory.newSymbol(value, type, yyline+1, yycolumn, yyline+1, yycolumn+yylength(), value);
  }
%}

LineTerminator = \r|\n|\r\n
InputCharacter = [^\r\n]
WhiteSpace     = {LineTerminator} | [ \t\f]

/* comments */
Comment = {TraditionalComment} | {EndOfLineComment}

TraditionalComment   = "/*" ~"*/" 
EndOfLineComment     = "//" {InputCharacter}* {LineTerminator}?
SMTLetter = [:letter:]
SMTLetterDigit = {SMTLetter} | [.'_] | [:digit:]
Identifier = {SMTLetter} {SMTLetterDigit}*

Numeral = 0 | [1-9][0-9]*

%state STRING PATTERN

%%

<YYINITIAL>  {
  "="             { return symbol(LexerSymbols.EQUALS); }
  "and"           { return symbol(LexerSymbols.AND); }
  "benchmark"     { return symbol(LexerSymbols.BENCHMARK); }
  "distinct"      { return symbol(LexerSymbols.DISTINCT); }
  "exists"        { return symbol(LexerSymbols.EXISTS); }
  "false"         { return symbol(LexerSymbols.FALSE); }
  "flet"          { return symbol(LexerSymbols.FLET); }

  "forall"        { return symbol(LexerSymbols.FORALL); }
  "if_then_else"  { return symbol(LexerSymbols.IF_THEN_ELSE); }
  "iff"           { return symbol(LexerSymbols.IFF); }
  "implies"       { return symbol(LexerSymbols.IMPLIES); }
  "ite"           { return symbol(LexerSymbols.ITE); }
  "let"           { return symbol(LexerSymbols.LET); }
  "logic"         { return symbol(LexerSymbols.LOGIC); }
  "not"           { return symbol(LexerSymbols.NOT); }
  "or"            { return symbol(LexerSymbols.OR); }
  "sat"           { return symbol(LexerSymbols.SAT); }

  "theory"        { return symbol(LexerSymbols.THEORY); }
  "true"          { return symbol(LexerSymbols.TRUE); }
  "unknown"       { return symbol(LexerSymbols.UNKNOWN); }
  "unsat"         { return symbol(LexerSymbols.UNSAT); }
  "xor"           { return symbol(LexerSymbols.XOR); }
  
  ":logic"        { return symbol(LexerSymbols.ATTR_LOGIC); } 
  ":assumption"   { return symbol(LexerSymbols.ATTR_ASSUMPTION); } 
  ":formula"      { return symbol(LexerSymbols.ATTR_FORMULA); } 
  ":status"       { return symbol(LexerSymbols.ATTR_STATUS); } 
  ":extrasorts"   { return symbol(LexerSymbols.ATTR_EXTRASORTS); } 
  ":extrafuns"    { return symbol(LexerSymbols.ATTR_EXTRAFUNS); } 
  ":extrapreds"   { return symbol(LexerSymbols.ATTR_EXTRAPREDS); } 
  ":notes"        { return symbol(LexerSymbols.ATTR_NOTES); } 
  ":pat"          { yybegin(PATTERN); return symbol(LexerSymbols.ATTR_PATTERN); }

  "("             { return symbol(LexerSymbols.LPAR); }
  ")"             { return symbol(LexerSymbols.RPAR); }
  "{"            { return symbol(LexerSymbols.LCPAR); }
  "}"            { return symbol(LexerSymbols.RCPAR); }

  /* Numbers, Ids and Strings */
  {Identifier}                   { return symbol(LexerSymbols.ID, yytext()); }
  {Identifier}\[{Numeral}(:{Numeral})*\]
  	                             { return symbol(LexerSymbols.INDEXED_ID, yytext()); }
  \?{Identifier}                  { return symbol(LexerSymbols.VAR, yytext().substring(1)); }
  \${Identifier}                  { return symbol(LexerSymbols.FVAR, yytext().substring(1)); }  
  :{Identifier}                  { return symbol(LexerSymbols.ATTRIBUTE, yytext().substring(1)); }
  
  [=<>&@#\+\-\*/%\|~]+               { return symbol(LexerSymbols.ARITH_SYMB, yytext()); }
  {Numeral}                      { return symbol(LexerSymbols.NUMERAL, yytext()); }
  {Numeral}\.[0-9]*              { return symbol(LexerSymbols.RATIONAL, yytext()); }
  \{([^\\\}\{]|\\\{|\\\}|\\\\)*\} { return symbol(LexerSymbols.USERVAL, yytext()); }
 
  \"                             { string.setLength(0); yybegin(STRING); }

 
  /* comments */
  {Comment}                      { /* ignore */ }
 
  /* whitespace */
  {WhiteSpace}                   { /* ignore */ }
}

<STRING> {
  \"                             { yybegin(YYINITIAL); 
                                   return symbol(LexerSymbols.ATTR_STRING, 
                                   string.toString()); }
  [^\n\r\"\\]+                   { string.append( yytext() ); }
  \\t                            { string.append('\t'); }
  \\n                            { string.append('\n'); }

  \\r                            { string.append('\r'); }
  \\\"                           { string.append('\"'); }
  \\                             { string.append('\\'); }
}

<PATTERN> {
  "{"            { return symbol(LexerSymbols.LCPAR); }
  "}"            { yybegin(YYINITIAL); return symbol(LexerSymbols.RCPAR); }
   "="             { return symbol(LexerSymbols.EQUALS); }
   "false"         { return symbol(LexerSymbols.FALSE); }
   "true"          { return symbol(LexerSymbols.TRUE); }
   "("             { return symbol(LexerSymbols.LPAR); }
   ")"             { return symbol(LexerSymbols.RPAR); }
   {Identifier}                   { return symbol(LexerSymbols.ID, yytext()); }
   \?{Identifier}                  { return symbol(LexerSymbols.VAR, yytext().substring(1)); }
   [=<>&@#\+\-\*/%\|~]+               { return symbol(LexerSymbols.ARITH_SYMB, yytext()); }
  {Numeral}                      { return symbol(LexerSymbols.NUMERAL, yytext()); }
  {Numeral}\.[0-9]*              { return symbol(LexerSymbols.RATIONAL, yytext()); }
  /* whitespace */
  {WhiteSpace}                   { /* ignore */ }
}

/* error fallback */
.|\n                             { return symbol(LexerSymbols.error, yytext()); }

<<EOF>>                          { return symbol(LexerSymbols.EOF); }
